/*global ol, map*/
var source = new ol.source.Vector();
var vector = new ol.layer.Vector({
    /*    shortTitle: 'Vector', */
        source: source,
        style: new ol.style.Style({
            fill: new ol.style.Fill({
                color: 'rgba(255, 255, 255, 0.2)'
            }),
            stroke: new ol.style.Stroke({
                color: '#ffcc33',
                width: 2
            }),
            image: new ol.style.Circle({
                radius: 7,
                fill: new ol.style.Fill({
                    color: '#ffcc33'
                })
            })
        })
    });
vector.setZIndex(0);
vector.setVisible(true);
map.addLayer(vector);
var modify = new ol.interaction.Modify({source: source});
map.addInteraction(modify);

var draw, snap; // global so we can remove them later
var measureTootlipsForClearing = [];

// Variables for measuring distance/area
var sketch;
var helpTooltipElement;
var helpTooltip;
var measureTooltipElement;
var measureTooltip;
var continuePolygonMsg = 'Click to continue drawing the polygon, double-click to end';
var continueLineMsg = 'Click to continue drawing the line, double-click to end';
var helpInitialMessage = 'Click to start drawing';

/**
 * Handle pointer move when drawing lines/polygons
 * @param {ol.MapBrowserEvent} evt The event.
 */
var pointerMoveHandler = function(evt) {
    if (evt.dragging) {
        return;
    }
    /** @type {string} */
    var helpMsg = helpInitialMessage;

    if (sketch) {
        var geom = (sketch.getGeometry());
        if (geom instanceof ol.geom.Polygon) {
            helpMsg = continuePolygonMsg;
        } else if (geom instanceof ol.geom.LineString) {
            helpMsg = continueLineMsg;
        }
    }

    helpTooltipElement.innerHTML = helpMsg;
    helpTooltip.setPosition(evt.coordinate);

    helpTooltipElement.classList.remove('hidden');
};


/**
 * Format length output.
 * @param {ol.geom.LineString} line The line.
 * @return {string} The formatted length.
 */
var formatLength = function(line) {
    var length = ol.Sphere.getLength(line);
    var output;
    if (length > 100) {
        output = (Math.round(length / 1000 * 100) / 100) +
            ' ' + 'km';
    } else {
        output = (Math.round(length * 100) / 100) +
            ' ' + 'm';
    }
    return output;
};

/**
 * Format area output.
 * @param {ol.geom.Polygon} polygon The polygon.
 * @return {string} Formatted area.
 */
var formatArea = function(polygon) {
    var area = ol.Sphere.getArea(polygon);
    var output;
    if (area > 10000) {
        output = (Math.round(area / 1000000 * 100) / 100) +
            ' ' + 'km<sup>2</sup>';
    } else {
        output = (Math.round(area * 100) / 100) +
            ' ' + 'm<sup>2</sup>';
    }
    return output;
};

/**
* Handle draw event.
*/
var setDraw = function (interaction_name) {
    "use strict";
    map.removeInteraction(draw);
    map.removeInteraction(snap);
    if (interaction_name === 'Clear') {
        source.clear();
        measureTootlipsForClearing.forEach(function (tootlip) {
           console.log(map.removeOverlay(tootlip));
        });
        console.log(map.removeOverlay(helpTooltip))
    } else { 
        addInteractions(interaction_name);
    }

    if (interaction_name === 'Polygon' || interaction_name === 'LineString') {
        registerDrawListeners(draw);
    }
};

var saveGPX = function () {
    var format = new ol.format.KML({extractStyles: false, extractAttributes:true});
    var result = format.writeFeatures(source.getFeatures(), {featureProjection: 'EPSG:3857' });
   // result = result.replace('<gpx xmlns="http://www.topografix.com/GPX/1/1">','<?xml version="1.0" encoding="UTF-8"?><gpx version="1.0" creator="BGMountains - https://kade.si/" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns="http://www.topografix.com/GPX/1/0" xsi:schemaLocation="http://www.topografix.com/GPX/1/0 http://www.topografix.com/GPX/1/0/gpx.xsd">');
    var blob = new Blob([result], {type: "text/plain;charset=utf-8;"});
    datetime = new Date().toISOString();
    var myFileName = 'layer-' + datetime + '.kml';
    saveAs(blob, myFileName);
    return false;
}

var saveFile = function (format, fname) {
    var result = format.writeFeatures(source.getFeatures(), {featureProjection: 'EPSG:3857' });
   // result = result.replace('<gpx xmlns="http://www.topografix.com/GPX/1/1">','<?xml version="1.0" encoding="UTF-8"?><gpx version="1.0" creator="BGMountains - https://kade.si/" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns="http://www.topografix.com/GPX/1/0" xsi:schemaLocation="http://www.topografix.com/GPX/1/0 http://www.topografix.com/GPX/1/0/gpx.xsd">');
    var blob = new Blob([result], {type: "text/plain;charset=utf-8;"});
    datetime = new Date().toISOString();
  //  var myFileName = 'layer-' + datetime + '.kml';
    saveAs(blob, fname);
    return false;
}

function addInteractions(interaction_name) {
    "use strict";
    draw = new ol.interaction.Draw({
        source: source,
        type: /** @type {ol.geom.GeometryType} */ (interaction_name)
    });
    map.addInteraction(draw);
    snap = new ol.interaction.Snap({source: source});
    map.addInteraction(snap);
}

/**
 * Creates a new help tooltip
 */
function createHelpTooltip() {
    if (helpTooltipElement) {
        helpTooltipElement.parentNode.removeChild(helpTooltipElement);
    }
    helpTooltipElement = document.createElement('div');
    helpTooltipElement.className = 'non-bootstrap-tooltip hidden';
    helpTooltip = new ol.Overlay({
        element: helpTooltipElement,
        offset: [15, 0],
        positioning: 'center-left'
    });
    map.addOverlay(helpTooltip);
}

/**
 * Creates a new measure tooltip
 */
function createMeasureTooltip() {
    if (measureTooltipElement) {
        measureTooltipElement.parentNode.removeChild(measureTooltipElement);
    }
    measureTooltipElement = document.createElement('div');
    measureTooltipElement.className = 'non-bootstrap-tooltip non-bootstrap-tooltip-measure';
    measureTooltip = new ol.Overlay({
        element: measureTooltipElement,
        offset: [0, -15],
        positioning: 'bottom-center'
    });
    map.addOverlay(measureTooltip);
    measureTootlipsForClearing.push(measureTooltip);
}

function registerDrawListeners(draw) {
    createMeasureTooltip();
    createHelpTooltip();

    var listener;

    draw.on('drawstart',
        function(evt) {
            sketch = evt.feature;

            /** @type {ol.Coordinate|undefined} */
            var tooltipCoord = evt.coordinate;

            listener = sketch.getGeometry().on('change', function(evt) {
                var geom = evt.target;
                var output;
                if (geom instanceof ol.geom.Polygon) {
                    output = formatArea(geom);
                    tooltipCoord = geom.getInteriorPoint().getCoordinates();
                } else if (geom instanceof ol.geom.LineString) {
                    output = formatLength(geom);
                    tooltipCoord = geom.getLastCoordinate();
                }
                measureTooltipElement.innerHTML = output;
                measureTooltip.setPosition(tooltipCoord);
            });
        }, this);

    draw.on('drawend',
        function() {
            measureTooltipElement.className = 'non-bootstrap-tooltip non-bootstrap-tooltip-static';
            measureTooltip.setOffset([0, -7]);
            sketch = null;
            measureTooltipElement = null; // Unset the tooltip so that a new one can be created
            createMeasureTooltip();
            helpTooltipElement.innerHTML = helpInitialMessage; // Reset the help message
            ol.Observable.unByKey(listener);
        }, this);


    // Register mouse pointer move listener only after draw tool click
    map.on('pointermove', pointerMoveHandler);

    // Hide help tooltip when mouse exits the map
    map.getViewport().addEventListener('mouseout', function() {
        helpTooltipElement.classList.add('hidden');
    });
}
